import React, { createContext, useContext, useState, useEffect } from "react";
import { auth, signInWithGooglePopup } from "../lib/firebase";
import {
    signInWithEmailAndPassword,
    createUserWithEmailAndPassword,
    signOut,
    onAuthStateChanged,
    updateProfile,
    sendPasswordResetEmail
} from "firebase/auth";
import * as api from "../lib/api";

const AuthContext = createContext();

export function useAuth() {
    return useContext(AuthContext);
}

export function AuthProvider({ children }) {
    const [user, setUser] = useState(null);
    const [userData, setUserData] = useState(null); // { role, approved }
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        const unsubscribe = onAuthStateChanged(auth, async (u) => {
            if (u) {
                // Get token and set it in API
                const token = await u.getIdToken();
                api.setAuthToken(token);

                // Fetch user data (role, approval)
                try {
                    const meta = await api.getAuthStatus();
                    setUserData(meta);
                } catch (error) {
                    console.error("Failed to fetch auth status", error);
                }

                setUser(u);
            } else {
                api.setAuthToken(null);
                setUser(null);
                setUserData(null);
            }
            setLoading(false);
        });

        return unsubscribe;
    }, []);

    const login = (email, password) => {
        return signInWithEmailAndPassword(auth, email, password);
    };

    const register = (email, password) => {
        return createUserWithEmailAndPassword(auth, email, password);
    };

    const logout = () => {
        return signOut(auth);
    };

    const signInWithGoogle = () => {
        return signInWithGooglePopup();
    };

    const updateUser = (name, photo) => {
        return updateProfile(auth.currentUser, { displayName: name, photoURL: photo });
    };

    const resetPassword = (email) => {
        return sendPasswordResetEmail(auth, email);
    };

    const value = {
        user,
        userData,
        login,
        register,
        logout,
        signInWithGoogle,
        updateUser,
        resetPassword,
        loading
    };

    return (
        <AuthContext.Provider value={value}>
            {!loading && children}
        </AuthContext.Provider>
    );
}
